#include "immanc.h"

					/* Lexicons */

struct Lexicon *initLexicon()
{
	struct Lexicon *lexicon;
	
	lexicon=(struct Lexicon *)malloc(sizeof(struct Lexicon));
	lexicon->activeCount=0;
	lexicon->totalCount=0;
	lexicon->head=NULL;
	return lexicon;
}

void freeLexEntry(struct LexEntry *entry)
{
	if (entry!=NULL) {
		freeLexEntry(entry->left);
		freeLexEntry(entry->right);
		free(entry->text);
		free(entry);
	}
}

void freeLexicon(struct Lexicon *lexicon)
{
	freeLexEntry(lexicon->head);
	free(lexicon);
}

int findLexEntry(struct LexEntry *entry, char *word)
/* This returns the index of the word if it's in the lexicon, zero if absent.
	(We have to look out for removed items by checking the sign of the index) */
{
	int position;
	
	if (entry==NULL)
		return 0;
	position=strcmp(word, entry->text);
	if (!position) {
		if (entry->index>0)
			return entry->index;
		else return 0;
	}
	else 
		if (position<0)
			return findLexEntry(entry->left, word);
	else return findLexEntry(entry->right, word);
}

int findLexWord(struct Lexicon *lexicon, char *word)
{
	return findLexEntry(lexicon->head, word);
}

struct LexEntry *initLexEntry(char *word, int n)
{
	struct LexEntry *entry;
	
	entry=(struct LexEntry *)malloc(sizeof(struct LexEntry));
	entry->text=(char *)malloc(strlen(word)+1);  // char added by GW
	memcpy(entry->text, word, strlen(word)+1);
	entry->index=n;
	entry->left=NULL;
	entry->right=NULL;
	return entry;
}

int getLexEntry(struct Lexicon *lexicon, struct LexEntry *entry, char *word)
/* This works like findLexEntry, except if the word is absent from the lexicon,
	it is added and the newly-generated entry's index is returned. (We also have
	to look out for removed words by checking the sign of the index.) */
{
	int position;
	
	if (lexicon->totalCount==0) {
		lexicon->totalCount=1;
		lexicon->activeCount=1;
		lexicon->head=initLexEntry(word, 1);
		return 1;
	}
	position=strcmp(word, entry->text);
	if (position==0) {
		if (entry->index<0) {
			entry->index= -entry->index;
			lexicon->activeCount++;
		}
		return entry->index;
	}
	if (position<0) {
		if (entry->left!=NULL) 
			return getLexEntry(lexicon, entry->left, word);
		else {
			lexicon->totalCount++;
			lexicon->activeCount++;
			entry->left=initLexEntry(word, lexicon->totalCount);
			return lexicon->totalCount;
		}
	}
	if (entry->right!=NULL)
		return getLexEntry(lexicon, entry->right, word);
	else {
		lexicon->totalCount++;
		lexicon->activeCount++;
		entry->right=initLexEntry(word, lexicon->totalCount);
		return lexicon->totalCount;
	}
}		

int getLexWord(struct Lexicon *lexicon, char *word)
{
	return getLexEntry(lexicon, lexicon->head, word);
}

int removeLexEntry(struct LexEntry *entry, char *word)
{
	int position;
	
	if (entry==NULL)
		return 0;
	position=strcmp(word, entry->text);
	if (!position) {
		entry->index= -entry->index;
		return entry->index;
	}
	else if (position<0)
		return removeLexEntry(entry->left, word);
	else return removeLexEntry(entry->right, word);
}

void removeLexWord(struct Lexicon *lexicon, char *word)
/* This does not remove the word from the lexicon; it only marks it with a
	minus sign for later recognition as removed. (The original version did
	actually remove the word, but also reassigned all indices, invalidating the
	associated network in a thesaurus) */
{
	if (removeLexEntry(lexicon->head, word))
		lexicon->activeCount--;
}

void printLexEntry(struct LexEntry *entry)
{
	if (entry->left!=NULL)
		printLexEntry(entry->left);
	if (entry->index>0)
		printf("%s\n", entry->text);
	if (entry->right!=NULL)
		printLexEntry(entry->right);
}

void printLexicon(struct Lexicon *lexicon)
{
	printLexEntry(lexicon->head);
}

void writeLexEntry(FILE *fout, struct LexEntry *entry)
{
	if (entry->left!=NULL)
		writeLexEntry(fout, entry->left);
	if (entry->index>0)
		fprintf(fout, "%s\n", entry->text);
	if (entry->right!=NULL)
		writeLexEntry(fout, entry->right);
}

void writeLexicon(struct Lexicon *lexicon, char *filename)
{
	FILE *fout;
	
	fout=fopen(filename, "w");
	writeLexEntry(fout, lexicon->head);
	fclose(fout);
}

int caseInsensitiveCompare(const void *arg1, const void *arg2)   // const added by GW to try and get listlexicon to work (below)
{
	char *string1, *string2;
	
	string1= *((char **)arg1);
	string2= *((char **)arg2);
	#ifdef GCC
	return strcasecmp(string1, string2);
        #endif
	#ifdef USPARC
       	return (int)strcasecmp(string1, string2);
        #endif
	#ifdef WINDOWS
       	return (int)stricmp(string1, string2);
        #endif
	#ifdef MAC
       	return (int)stricmp(string1, string2);
	#endif
}

char **listLexicon(struct Lexicon *lexicon)
{
	int i, j;
	char **list, *word;
	
	j=0;
	list=(char **)malloc((lexicon->activeCount)*sizeof(char*));
	for (i=1; i<=lexicon->totalCount; i++)
		if (word=lexWord(lexicon, i))
			list[j++]=word;
        #ifdef MAC
	qsort(list,j,sizeof( char * ),(_Cmpfun *) caseInsensitiveCompare);
        #else
        qsort(list,j,sizeof( char * ),caseInsensitiveCompare);
        #endif
	return list;
}

char *lexItemNode(struct LexEntry *entry, int n)
{
	char *answer;
	
	if (entry==NULL)
		return NULL;
	if (n==entry->index)
		return entry->text;
	else if (-n==entry->index)
		return NULL;
	answer=lexItemNode(entry->left, n);
	if (answer==NULL)
		answer=lexItemNode(entry->right, n);
	return answer;
}

char *lexWord(struct Lexicon *lexicon, int n)
{
	return lexItemNode(lexicon->head, n);
}

int totalWords(struct Lexicon *lexicon)
{
	return lexicon->totalCount;
}

int activeWords(struct Lexicon *lexicon)
{
	return lexicon->activeCount;
}

